/*
 * Copyright (c) 2014 NIBIO <http://www.nibio.no/>. 
 * 
 * This file is part of AppleScabModel.
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 * 
 */
package no.nibio.vips.model.applescabmodel;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;
import no.nibio.vips.entity.ModelConfiguration;
import no.nibio.vips.entity.Result;
import no.nibio.vips.entity.ResultImpl;
import no.nibio.vips.entity.WeatherObservation;
import no.nibio.vips.i18n.I18nImpl;
import no.nibio.vips.model.ConfigValidationException;
import no.nibio.vips.model.Model;
import no.nibio.vips.model.ModelExcecutionException;
import no.nibio.vips.model.ModelId;
import no.nibio.vips.util.CommonNamespaces;
import no.nibio.vips.util.InvalidAggregationTypeException;
import no.nibio.vips.util.WeatherElements;
import no.nibio.vips.util.WeatherObservationListException;
import no.nibio.vips.util.WeatherUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.core.type.TypeReference;
import java.io.IOException;
import no.nibio.vips.util.ModelUtil;

/**
 * This model is based on the old one in VIPS
 * @copyright 2013 <a href="http://www.nibio.no/">NIBIO</a>
 * @author Tor-Einar Skog <tor-einar.skog@nibio.no>
 */
public class AppleScabModel extends I18nImpl implements Model{
    
    /**
     * The unique identifier for this model
     */
    public final static ModelId MODEL_ID = new ModelId("APPLESCABM");
    
    /* 
     * Config parameters
     */
    // If total precipitation is lower than this, we have "dry day" conditions
    private final static Double DRY_DAY_PRECIPITATION_THRESHOLD = 0.2;
    // If total leaf wetness is lower than this, we have "dry day" conditions
    private final static Double DRY_DAY_LEAF_WETNESS_THRESHOLD = 10d;
    // If number of consecutive dry days exceeds this, we have dry conditions (no ascospore maturity development)
    // See calculateAccumulatedTemperature()
    private final static Integer DRY_DAYS_TEMPERATURE_ACCUMULATION_THRESHOLD = 7;
    
    // Used in accumulateMills()
    private final static Integer MAX_LEAF_DRYING_TIME = 8;
    private final static Double LEAF_WETNESS_THRESHOLD = 20d;
    private final static Integer APPLE_SCAB_STADIUM_ASCOSPORE = 1;
    private final static Integer APPLE_SCAB_STADIUM_CONIDIA = 2;
    
    // Thresholds for warning status
    private final static Double MILLS_THRESHOLD_RED_WARNING = 100d;
    private final static Double MILLS_THRESHOLD_YELLOW_WARNING = 90d;

    // The data store for all calculations and data
    private AppleScabCalculations calculations;
    // What timezone is the calculation for
    private TimeZone timeZone;
    // When do we start calculation ascospore maturity
    // Normally the date for "green tip"
    private Date startDateAscosporeMaturity;
    
    // Weather data collections
    private List<WeatherObservation> TM; // Mean temperature
    private List<WeatherObservation> RR; // Rainfall
    private List<WeatherObservation> BT; // Leaf wetness
    
    // Helpers for calculation if missing BT
    private List<WeatherObservation> Q0; // Global radiation
    private List<WeatherObservation> FM2; // Wind speed 2m above ground (m/s)
    private List<WeatherObservation> UM; // Relative humidity
    
    // Helper class
    private final WeatherUtil weatherUtil;
    private final ModelUtil modelUtil;
    
    public AppleScabModel()
    {
        // Setting the file name of the resource bundle
        super("no.nibio.vips.model.applescabmodel.texts");
        this.weatherUtil = new WeatherUtil();
        this.modelUtil = new ModelUtil();
    }

    /**
     * Executes the model and returns result
     * @return
     * @throws ModelExcecutionException 
     */
    @Override
    public List<Result> getResult() throws ModelExcecutionException {
        try {
            this.accumulateMills();
        } catch (WeatherObservationListException ex) {
            Logger.getLogger(AppleScabModel.class.getName()).log(Level.SEVERE, null, ex);
            throw new ModelExcecutionException(ex.getMessage());
        }
        //System.out.println(this.calculations.toString());
        // Looping through background data, adding as much as possible
        List<Result> results = new ArrayList<>();
        for(Date timeStamp:this.calculations.getSortedDateKeys())
        {
            Result result = new ResultImpl();
            result.setValidTimeStart(timeStamp);
            
            result.setValue(CommonNamespaces.NS_FORECAST, "THRESHOLD_LOW", MILLS_THRESHOLD_YELLOW_WARNING.toString());
            result.setValue(CommonNamespaces.NS_FORECAST, "THRESHOLD_HIGH", MILLS_THRESHOLD_RED_WARNING.toString());
            
            result.setValue(this.getModelId().toString(), AppleScabCalculations.ASCOSPORE_MATURITY, this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.ASCOSPORE_MATURITY) != null ? String.valueOf(this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.ASCOSPORE_MATURITY)) : "");
            result.setValue(this.getModelId().toString(), AppleScabCalculations.ACCUMULATED_MILLS, String.valueOf(this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.ACCUMULATED_MILLS)));
            result.setValue(CommonNamespaces.NS_WEATHER, AppleScabCalculations.TM, String.valueOf(this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.TM)));
            result.setValue(CommonNamespaces.NS_WEATHER, AppleScabCalculations.BT, String.valueOf(this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.BT)));
            result.setValue(CommonNamespaces.NS_WEATHER, AppleScabCalculations.RR, String.valueOf(this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.RR)));
            Double accumulatedTemperature = this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.ACCUMULATED_TEMPERATURE);
            result.setValue(CommonNamespaces.NS_WEATHER, AppleScabCalculations.ACCUMULATED_TEMPERATURE, accumulatedTemperature != null ? String.valueOf(accumulatedTemperature) : "");
            Double aggregatedLeafWetness = this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.AGGREGATED_LEAF_WETNESS);
            result.setValue(CommonNamespaces.NS_WEATHER, AppleScabCalculations.AGGREGATED_LEAF_WETNESS, aggregatedLeafWetness != null ? String.valueOf(aggregatedLeafWetness) : "");
            Double aggregatedTemperature = this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.AGGREGATED_TEMPERATURE);
            result.setValue(CommonNamespaces.NS_WEATHER, AppleScabCalculations.AGGREGATED_TEMPERATURE, aggregatedTemperature != null ? String.valueOf(aggregatedTemperature) : "");
            Double aggregatedPrecipitation = this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.AGGREGATED_PRECIPITATION);
            result.setValue(CommonNamespaces.NS_WEATHER, AppleScabCalculations.AGGREGATED_PRECIPITATION, aggregatedPrecipitation != null ? String.valueOf(aggregatedPrecipitation) : "");
            
            // Deciding warning status
            result.setWarningStatus(this.getWarningStatus(timeStamp));
            
            results.add(result);
        }
        return results;
    }
    
    /**
     * 
     * @param timeStamp
     * @return 
     */
    private Integer getWarningStatus(Date timeStamp)
    {
        Double accumulatedMills = this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.ACCUMULATED_MILLS);
        if(accumulatedMills == null)
        {
            return Result.WARNING_STATUS_NO_WARNING;
        }
        // If ascospore maturity == 100%, primary infection period is over. So no warning issued after that
        else if(this.calculations.getParamDoubleValueForDate(this.weatherUtil.normalizeToExactDate(timeStamp, this.timeZone), AppleScabCalculations.ASCOSPORE_MATURITY) == 100d)
        {
            return Result.WARNING_STATUS_NO_WARNING;
        }
        else
        {
            if(accumulatedMills >= MILLS_THRESHOLD_RED_WARNING)
            {
                return Result.WARNING_STATUS_HIGH_RISK;
            }
            else if(accumulatedMills >= MILLS_THRESHOLD_YELLOW_WARNING)
            {
                return Result.WARNING_STATUS_MINOR_RISK;
            }
            else
            {
                return Result.WARNING_STATUS_NO_RISK;
            }
        }
    }

    @Override
    public ModelId getModelId() {
        return MODEL_ID;
    }

    @Override
    public String getModelName() {
        return this.getModelName(Model.DEFAULT_LANGUAGE);
    }

    @Override
    public String getModelName(String language) {
        return this.getText("name", language);
    }

    @Override
    public String getModelDescription() {
        return this.getModelDescription(Model.DEFAULT_LANGUAGE);
    }

    @Override
    public String getModelDescription(String language) {
        try
        {
            return this.modelUtil.getTextWithBase64EncodedImages(this.getText("description", language), this.getClass());
        }
        catch(IOException ex)
        {
            return this.getText("description", language);
        }
    }

    @Override
    public String getModelUsage() {
        return this.getModelUsage(Model.DEFAULT_LANGUAGE);
    }

    @Override
    public String getModelUsage(String language) {
        return this.getText("usage", language);
    }

    @Override
    public String getSampleConfig() {
        return  "{\n" +
                "\t\"loginInfo\":{\n" +
                "\t\t\"username\":\"example\",\n" +
                "\t\t\"password\":\"example\"\n" +
                "\t},\n" +
                "\t\"modelId\":\"" + MODEL_ID.toString() + "\",\n" +
                "\t\"configParameters\":{\n" +
                "\t\t\"timeZone\":\"Europe/Oslo\",\n" +
                "\t\t\"startDateAscosporeMaturity\":\"2012-03-25\",\n" +
                "\t\t\"observations\":[\n" +
                "\t\t{\n" +
                "\t\t\t\t\"timeMeasured\": \"2012-03-25T00:00:00+02:00\",\n" +
                "\t\t\t\t\"elementMeasurementTypeId\":\"TM\",\n" +
                "\t\t\t\t\"logIntervalId\":1,\n" +
                "\t\t\t\t\"value\":2.937\n" +
                "\t\t},\n" +
                "\t\t{\n" +
                "\t\t\t\t\"timeMeasured\": \"2012-03-25T00:00:00+02:00\",\n" +
                "\t\t\t\t\"elementMeasurementTypeId\":\"RR\",\n" +
                "\t\t\t\t\"logIntervalId\":1,\n" +
                "\t\t\t\t\"value\":0\n" +
                "\t\t},\n" +
                "\t\t{\n" +
                "\t\t\t\t\"timeMeasured\": \"2012-03-25T00:00:00+02:00\",\n" +
                "\t\t\t\t\"elementMeasurementTypeId\":\"BT\",\n" +
                "\t\t\t\t\"logIntervalId\":1,\n" +
                "\t\t\t\t\"value\":0\n" +
                "\t\t}\n" +
                "\t\t]\n" +
                "\t}\n" +
                "}\n";
    }

    /**
     * Input parameters:
     * <ul>
     * <li>startDateAscosporeMaturity {@see java.util.Date} - the date from which to start calculating ascospore maturity</li>
     * <li>timeZone {@see java.lang.String}</li> - Id for timezone in which you want to work
     * <li>observations {@see java.util.List} of {@see no.nibio.vips.entity.WeatherObservation}</li>
     * 
     * </ul>
     * 
     */
    @Override
    public void setConfiguration(ModelConfiguration config) throws ConfigValidationException {
        // TODO: Validate all input!!!!
        initCollections();
        ObjectMapper mapper = new ObjectMapper();
        
        // Setting timezone
        this.timeZone = TimeZone.getTimeZone((String) config.getConfigParameter("timeZone"));
        
        //this.startDateAscosporeMaturity = mapper.convertValue(config.getConfigParameter("startDateAscosporeMaturity"), new TypeReference<java.util.Date>(){});
        String[] dateparts;
        try
        {
            dateparts = config.getConfigParameter("startDateAscosporeMaturity").toString().split("-");
        }
        catch (NullPointerException ex){
            throw new ConfigValidationException("startDateAscosporeMaturity not set");
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        format.setTimeZone(timeZone);
        try
        {
          
            this.startDateAscosporeMaturity = format.parse(config.getConfigParameter("startDateAscosporeMaturity").toString());
        }
        catch(ParseException | NullPointerException ex)
        {
            throw new ConfigValidationException("Illegal date format for startDateAscosporeMaturity: " + config.getConfigParameter("startDateAscosporeMaturity"));
        }
        
        // ############ Weather data ##############
        // Getting weather data, validating
        this.calculations = new AppleScabCalculations();
        List<WeatherObservation> observations = mapper.convertValue(config.getConfigParameter("observations"), new TypeReference<List<WeatherObservation>>(){});

        for(WeatherObservation o:observations)
        {
            switch(o.getElementMeasurementTypeId())
            {
                case WeatherElements.TEMPERATURE_MEAN:
                    this.TM.add(o); 
                    break;
                case WeatherElements.PRECIPITATION:
                    this.RR.add(o); 
                    break;
                case WeatherElements.LEAF_WETNESS_DURATION:
                    this.BT.add(o); 
                    break;
                default:
                    // Let it pass in silence 
                    break;
            }
        }
        
        // Basic time series health check
        WeatherUtil wUtil = new WeatherUtil();
        
        try
        {
            wUtil.isHomogenousTimeSeries(this.TM, this.RR, this.BT);
        }
        catch(WeatherObservationListException ex)
        {
            throw new ConfigValidationException(ex.getMessage());
        }
        
        // If all that's OK, chop of all hour values exceeding the last full day
        // of hourly data
        this.TM = wUtil.cutTrailingHourlyValues(this.TM, timeZone);
        this.BT = wUtil.cutTrailingHourlyValues(this.BT, timeZone);
        this.RR = wUtil.cutTrailingHourlyValues(this.RR, timeZone);
        
        // Must be minimum 24 hours
        int minimumNumberOfWeatherData = 24;
        if((this.TM.size() + this.RR.size() + this.BT.size()) / 3 < minimumNumberOfWeatherData)
        {
            throw new ConfigValidationException("Minimum number of weather data = " + minimumNumberOfWeatherData);
        }
        
        // Then, at last, add to calculations
        // Weather data are OK, let's add them to calculations matrix
        this.calculations = new AppleScabCalculations();
        for(WeatherObservation o: this.TM)
        {
            this.calculations.setParamDoubleValueForDate(o.getTimeMeasured(), AppleScabCalculations.TM, o.getValue());
        }
        for(WeatherObservation o: this.RR)
        {
            this.calculations.setParamDoubleValueForDate(o.getTimeMeasured(), AppleScabCalculations.RR, o.getValue());
        }
        for(WeatherObservation o: this.BT)
        {
            this.calculations.setParamDoubleValueForDate(o.getTimeMeasured(), AppleScabCalculations.BT, o.getValue());
        }
        
        
        
        // the date for which to start calculation of ascospore maturity must be in the range of weather data
        if(this.startDateAscosporeMaturity.compareTo(this.calculations.getFirstDateWithValues()) < 0
                || this.startDateAscosporeMaturity.compareTo(this.calculations.getLastDateWithValues()) > 0
                )
        {
            throw new ConfigValidationException("Start date for calculation (" + this.startDateAscosporeMaturity + ") is outside of range for weather data (" + this.calculations.getFirstDateWithValues() + "-" + this.calculations.getLastDateWithValues() + ")");
        }
    }
    
    /**
     * Loops through dates, calculates Mill's value for each day and accumulates
     * @throws WeatherObservationListException 
     */
    private void accumulateMills() throws WeatherObservationListException
    {
        Double accumulatedMills = 0d;
        Integer currentAppleScabStadium = APPLE_SCAB_STADIUM_ASCOSPORE;
        Double ascosporeMaturity = 0d;
        AscosporeMaturityTable ascosporeMaturityTable = new AscosporeMaturityTable();
        MillsTable millsTable = new MillsTable();
        Date currentTimeStamp = this.startDateAscosporeMaturity;
        Collections.sort(this.TM);
        Date endTimeStamp = this.TM.get(this.TM.size()-1).getTimeMeasured();
        Calendar cal = Calendar.getInstance(this.timeZone);
        cal.setTime(currentTimeStamp);
        Integer hoursSinceLeafWetness = 0;
        while(currentTimeStamp.compareTo(endTimeStamp) <= 0)
        {
            if(ascosporeMaturity >= 100)
            {
                currentAppleScabStadium = APPLE_SCAB_STADIUM_CONIDIA;
            }
            // We update and store the ascospore maturity only at midnight, when we have an updated accumulated temperature
            if(currentTimeStamp.equals(this.weatherUtil.normalizeToExactDate(currentTimeStamp, this.timeZone)))
            {
                ascosporeMaturity = ascosporeMaturityTable.getAscosporeMaturity(this.getAccumulatedTemperature(currentTimeStamp));
                this.calculations.setParamDoubleValueForDate(currentTimeStamp, AppleScabCalculations.ASCOSPORE_MATURITY, ascosporeMaturity);
            }
            
            // We accumulate Mills only when there are humid conditions
            if(this.getLeafWetness(currentTimeStamp) != null && this.getLeafWetness(currentTimeStamp) < LEAF_WETNESS_THRESHOLD)
            {
                hoursSinceLeafWetness++;
                // If dry for long enough, reset accumulated Mills
                if(hoursSinceLeafWetness >= MAX_LEAF_DRYING_TIME)
                {
                    accumulatedMills = 0d;
                }
            }
            else
            {
                hoursSinceLeafWetness = 0;
                if(this.getTemperature(currentTimeStamp) != null)
                {
                    accumulatedMills += millsTable.getMillsValue(this.getTemperature(currentTimeStamp), currentAppleScabStadium);
                }
            }

            this.calculations.setParamDoubleValueForDate(currentTimeStamp, AppleScabCalculations.ACCUMULATED_MILLS, accumulatedMills);
            
            cal.add(Calendar.HOUR, 1);
            currentTimeStamp = cal.getTime();
        }

    }
    
    public Double getTemperature(Date timeStamp)
    {
        return this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.TM);
    }
    
    public Double getLeafWetness(Date timeStamp)
    {
        return this.calculations.getParamDoubleValueForDate(timeStamp, AppleScabCalculations.BT);
    }

    
    /**
     * 
     * @param atDate
     * @return 
     */
   public Double calculateAscosporeMaturity(Date atDate) throws WeatherObservationListException
   {
       
       Double accumulatedTemperature = this.getAccumulatedTemperature(atDate);
       return this.ascosporeValue(accumulatedTemperature);
  
   }

   /**
    * @param 
    *  @return percent mature ascospore
    */
   protected Double ascosporeValue( Double accumulatedTemperature )
   {
      return new AscosporeMaturityTable().getAscosporeMaturity(accumulatedTemperature);
   }


   

   /**
    * Fetches accumulated temperature at a given Date
    * @param atDate
    * @return
    * @throws WeatherObservationListException 
    */
    protected Double getAccumulatedTemperature(Date atDate) throws WeatherObservationListException {
        if(
            this.calculations == null 
            || 
            this.calculations.getParamDoubleValueForDate(
                this.startDateAscosporeMaturity, AppleScabCalculations.ACCUMULATED_TEMPERATURE
            ) == null
        )
        {
            // TODO: Add aggregates of the humidity values
            this.calculateAccumulatedTemperature();
        }
                
        // We must normalize date to midnight
        atDate = this.weatherUtil.normalizeToExactDate(atDate, this.timeZone);
        
        return this.calculations.getParamDoubleValueForDate(
            atDate, AppleScabCalculations.ACCUMULATED_TEMPERATURE
        );
    }

    /**
     * 
     * Calculates accumulated temperature, stores per date for lookup
     * @throws WeatherObservationListException 
     */
    private void calculateAccumulatedTemperature() throws WeatherObservationListException{
        if(! isAccumulatedTemperatureCalculationBackgroundDataReady())
        {
                aggregateWeatherObservations();
        }

        // We start at first date for calculation
        Double accumulatedTemperature = 0d;
        // Assuming dry conditions at start of calculations (approximation/simplification!!!)
        Integer consecutiveDryDays = DRY_DAYS_TEMPERATURE_ACCUMULATION_THRESHOLD; 
        //Integer consecutiveDryDays = 0;
        for(Date currentDate:this.calculations.getSortedDateAtMidnightKeys(this.timeZone))
        {
            //System.out.println("currentDate=" + currentDate);
            if(currentDate.compareTo(this.startDateAscosporeMaturity) >= 0)
            {
                Double leafWetness = this.calculations.getParamDoubleValueForDate(currentDate, AppleScabCalculations.AGGREGATED_LEAF_WETNESS);
                Double precipitation = this.calculations.getParamDoubleValueForDate(currentDate, AppleScabCalculations.AGGREGATED_PRECIPITATION);
                Double temperature = this.calculations.getParamDoubleValueForDate(currentDate, AppleScabCalculations.AGGREGATED_TEMPERATURE);
                
                if(leafWetness < DRY_DAY_LEAF_WETNESS_THRESHOLD || precipitation < DRY_DAY_PRECIPITATION_THRESHOLD)
                {
                    consecutiveDryDays++;
                }
                // If we have a "wet" day, reset counter
                else
                {
                    consecutiveDryDays = 0;
                }
                if(temperature > 0 && consecutiveDryDays <= DRY_DAYS_TEMPERATURE_ACCUMULATION_THRESHOLD)
                {
                    accumulatedTemperature += temperature;
                }
                this.calculations.setParamDoubleValueForDate(currentDate, AppleScabCalculations.ACCUMULATED_TEMPERATURE, accumulatedTemperature);
            }
        }        
    }
    
    /**
     * Quick check to see if background data is ready
     * @return 
     */
    private boolean isAccumulatedTemperatureCalculationBackgroundDataReady()
    {
        if(this.calculations == null)
            return false;
        
        return (
            this.calculations.getParamDoubleValueForDate(this.startDateAscosporeMaturity, AppleScabCalculations.AGGREGATED_TEMPERATURE) != null
            && this.calculations.getParamDoubleValueForDate(this.startDateAscosporeMaturity, AppleScabCalculations.AGGREGATED_LEAF_WETNESS) != null
            && this.calculations.getParamDoubleValueForDate(this.startDateAscosporeMaturity, AppleScabCalculations.AGGREGATED_PRECIPITATION) != null
        );
         
    }
    
    /**
     * Sets up the collections
     */
    private void initCollections(){
        this.BT = new ArrayList();
        this.RR = new ArrayList();
        this.TM = new ArrayList();
        this.UM = new ArrayList();
        this.Q0 = new ArrayList();
        this.FM2 = new ArrayList();
    }
    
    /**
     * Takes the hourly values in BT, RR and TM and aggregates, places in DateMap
     * @throws WeatherObservationListException 
     */
    private void aggregateWeatherObservations() throws WeatherObservationListException
    {
        this.aggregateWeatherObservation(this.TM, AppleScabCalculations.AGGREGATED_TEMPERATURE, WeatherUtil.AGGREGATION_TYPE_AVERAGE);
        this.aggregateWeatherObservation(this.BT, AppleScabCalculations.AGGREGATED_LEAF_WETNESS, WeatherUtil.AGGREGATION_TYPE_SUM);
        this.aggregateWeatherObservation(this.RR, AppleScabCalculations.AGGREGATED_PRECIPITATION, WeatherUtil.AGGREGATION_TYPE_SUM);
    }
    
    /**
     * 
     * @param hourlyValues
     * @param dateMapKey the Key for this aggregate in the {@see AppleScabCalculations}
     * @param aggregationType average, sum, max, min. {@see WeatherUtil} for ids
     * @throws WeatherObservationListException 
     */
    private void aggregateWeatherObservation(List<WeatherObservation> hourlyValues, String dateMapKey, int aggregationType) throws WeatherObservationListException
    {
        if(this.calculations == null)
        {
            this.calculations = new AppleScabCalculations();
        }
        Collections.sort(this.TM);
        WeatherUtil wUtil = new WeatherUtil();
        List<WeatherObservation> dailyAggregate = new ArrayList<>();
        try {
            dailyAggregate = wUtil.getAggregatedDailyValues(hourlyValues, this.timeZone, 21, aggregationType,1);
        } 
        catch (InvalidAggregationTypeException ex) {
            // TODO
        } 
        
        if(dailyAggregate.isEmpty())
        {
            throw new WeatherObservationListException("No aggregated values for parameter " + hourlyValues.get(0).getElementMeasurementTypeId());
        }
        
        for(WeatherObservation observation:dailyAggregate)
        {
            if(observation.getTimeMeasured().compareTo(this.startDateAscosporeMaturity) >= 0)
            {
                this.calculations.setParamDoubleValueForDate(observation.getTimeMeasured(), dateMapKey, observation.getValue());
            }
        }   
    }

    @Override
    public String getLicense() {
        return 
                "  Copyright (c) 2014 NIBIO <http://www.nibio.no/>. \n" +
                "  \n" +
                "  This file is part of AppleScabModel.\n" +
                "  This program is free software: you can redistribute it and/or modify\n" +
                "  it under the terms of the GNU Affero General Public License as published by\n" +
                "  the Free Software Foundation, either version 3 of the License, or\n" +
                "  (at your option) any later version.\n" +
                " \n" +
                "  This program is distributed in the hope that it will be useful,\n" +
                "  but WITHOUT ANY WARRANTY; without even the implied warranty of\n" +
                "  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n" +
                "  GNU Affero General Public License for more details.\n" +
                " \n" +
                "  You should have received a copy of the GNU Affero General Public License\n" +
                "  along with this program.  If not, see <https://www.gnu.org/licenses/>.\n" +
                "  \n" ;
    }

    @Override
    public String getCopyright() {
        return "(c) 2014 NIBIO (http://www.nibio.no/). Contact: post@nibio.no";
    }
    
    @Override
    public String getWarningStatusInterpretation() {
        return this.getWarningStatusInterpretation(Model.DEFAULT_LANGUAGE);
    }

    @Override
    public String getWarningStatusInterpretation(String language) {
        return this.getText("statusInterpretation", language);
    }
}
