/*
 * Copyright (c) 2014 NIBIO <http://www.nibio.no/>. 
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */

package no.nibio.vips.logic.controller.servlet;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import org.apache.commons.fileupload2.core.DiskFileItemFactory;
import org.apache.commons.fileupload2.core.FileItem;
import org.apache.commons.fileupload2.core.FileUploadException;
import org.apache.commons.fileupload2.jakarta.servlet6.JakartaServletFileUpload;
import jakarta.ejb.EJB;
import jakarta.persistence.EntityManager;
import jakarta.persistence.NoResultException;
import jakarta.persistence.PersistenceContext;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import no.nibio.vips.logic.controller.session.MessageBean;
import no.nibio.vips.logic.controller.session.UserBean;
import no.nibio.vips.logic.entity.Message;
import no.nibio.vips.logic.entity.MessageLocale;
import no.nibio.vips.logic.entity.MessageTag;
import no.nibio.vips.logic.entity.VipsLogicUser;
import no.nibio.vips.logic.i18n.SessionLocaleUtil;
import no.nibio.vips.logic.messaging.MessagingBean;
import no.nibio.vips.logic.util.Globals;
import no.nibio.vips.logic.util.SystemTime;
import no.nibio.vips.util.ExceptionUtil;
import no.nibio.vips.util.ServletUtil;
import no.nibio.web.forms.FormUtil;
import no.nibio.web.forms.FormValidation;
import no.nibio.web.forms.FormValidationException;
import no.nibio.web.forms.FormValidator;

/**
 * @copyright 2014-2022 <a href="http://www.nibio.no/">NIBIO</a>
 * @author Tor-Einar Skog <tor-einar.skog@nibio.no>
 */
public class MessageController extends HttpServlet {
   
    @PersistenceContext(unitName="VIPSLogic-PU")
    EntityManager em;
    
    @EJB
    MessageBean messageBean;
    
    @EJB
    UserBean userBean;
    
    @EJB
    MessagingBean messagingBean;
    
    /** 
     * Processes requests for both HTTP <code>GET</code> and <code>POST</code> methods.
     * @param request servlet request
     * @param response servlet response
     * @throws ServletException if a servlet-specific error occurs
     * @throws IOException if an I/O error occurs
     */
    protected void processRequest(HttpServletRequest request, HttpServletResponse response)
    throws ServletException, IOException {
        response.setContentType("text/html;charset=UTF-8");
        String action = request.getParameter("action");
        VipsLogicUser user = (VipsLogicUser) request.getSession().getAttribute("user");
        // Default: View message list
        // for everyone
        if(action == null)
        {
            // Default date filter is one year back and one year ahead (related to SystemTime.getSystemTime())
            Calendar cal = Calendar.getInstance();
            cal.setTime(SystemTime.getSystemTime());
            cal.add(Calendar.YEAR, -1);
            Date datePubStart = cal.getTime();
            cal.add(Calendar.YEAR, 2);
            Date datePubEnd = cal.getTime();
            // Default tag filter is no tag filter
            
            List<Message> messages;
            if(user.isSuperUser())
            {
                messages = messageBean.getFilteredMessages(null,null,datePubStart, datePubEnd);
            }
            else
            {
                messages = messageBean.getFilteredMessages(user.getOrganizationId().getOrganizationId(),null, datePubStart, datePubEnd);
            }
            request.setAttribute("messages", messages);
            request.setAttribute("selectedMessageTags", em.createNamedQuery("MessageTag.findAll", MessageTag.class).getResultList());
            request.setAttribute("notSelectedMessageTags", new ArrayList<MessageTag>());
            request.setAttribute("datePubStart", datePubStart);
            request.setAttribute("datePubEnd", datePubEnd);
            
            // If this is a redirect from a controller, with a message to be passed on
            request.setAttribute("messageKey", request.getParameter("messageKey"));
            request.getRequestDispatcher("/messageList.ftl").forward(request, response);
        }
        else if(action.equals("filter"))
        {
            // Handling filters
            // Message tags
            // If no tags are selected, ALL tags are selected
            List<Integer> selectedMessageTags = request.getParameterValues("filterMessageTags") != null ? 
                                        FormUtil.getIdsFromMultipleSelect(request.getParameterValues("filterMessageTags"))
                                        : messageBean.getAllMessageTagIds();
            Map<String, List<MessageTag>> messageTags = messageBean
                    .getFilterMessageTags(selectedMessageTags);
            request.setAttribute("selectedMessageTags", messageTags.get("selected"));
            request.setAttribute("notSelectedMessageTags", messageTags.get("notSelected"));
            
            // Dates
            Date datePubStart, datePubEnd;
            try
            {
                datePubStart = FormUtil.getDate(request.getParameter("datePubStart"));
                datePubEnd = FormUtil.getDate(request.getParameter("datePubEnd"));
            }
            catch(ParseException ex)
            {
                Calendar cal = Calendar.getInstance();
                cal.setTime(SystemTime.getSystemTime());
                cal.add(Calendar.YEAR, -1);
                datePubStart = cal.getTime();
                cal.add(Calendar.YEAR, 2);
                datePubEnd = cal.getTime();
            }
            List<Message> messages;
            if(user.isSuperUser())
            {
                messages = messageBean.getFilteredMessages(null,selectedMessageTags, datePubStart, datePubEnd);
            }
            else
            {
                messages = messageBean.getFilteredMessages(user.getOrganizationId().getOrganizationId(),selectedMessageTags, datePubStart, datePubEnd);
            }
            request.setAttribute("datePubStart", datePubStart);
            request.setAttribute("datePubEnd", datePubEnd);
            request.setAttribute("messages", messages);
            
            // If this is a redirect from a controller, with a message to be passed on
            request.setAttribute("messageKey", request.getParameter("messageKey"));
            request.getRequestDispatcher("/messageList.ftl").forward(request, response);
        }
        else if(action.equals("editMessageForm"))
        {
            try
            {
                Integer messageId = Integer.valueOf(request.getParameter("messageId"));
                Message message = em.createNamedQuery("Message.findByMessageId", Message.class).setParameter("messageId", messageId).getSingleResult();
                request.setAttribute("message", message);
                String messageLocale = request.getParameter("messageLocale") != null ? 
                                request.getParameter("messageLocale") 
                                : SessionLocaleUtil.getCurrentLocale(request).getLanguage();
                request.setAttribute("messageLocale", messageLocale);
                // If whe have existing versions of the message in other languages, 
                // we must find the most relevant translation and use as reference
                if(message.getMessageLocaleSet().size() > 1)
                {
                    // Need to determine the most likely reference translation
                    MessageLocale referenceTranslationLocale = null;
                    // 1) the user's preferred language? Only if it's not already selected as current message language
                    if(!messageLocale.equals(SessionLocaleUtil.getCurrentLocale(request).getLanguage()))
                    {
                        // Check that it exists among the translations
                        for(MessageLocale ml : message.getMessageLocaleSet())
                        {
                            if(ml.getMessageLocalePK().getLocale().equals(SessionLocaleUtil.getCurrentLocale(request).getLanguage()))
                            {
                                referenceTranslationLocale = ml;
                            }
                        }
                    }
                    // 2) English
                    if(referenceTranslationLocale == null)
                    {
                        // Check that English translation exists
                        for(MessageLocale ml : message.getMessageLocaleSet())
                        {
                            if(ml.getMessageLocalePK().getLocale().equals("en"))
                            {
                                referenceTranslationLocale = ml;
                            }
                        }
                    }
                    // 3) The first other translation available
                    if(referenceTranslationLocale == null)
                    {
                        for(MessageLocale ml : message.getMessageLocaleSet())
                        {
                            if(!ml.getMessageLocalePK().getLocale().equals(messageLocale))
                            {
                                referenceTranslationLocale = ml;
                            }
                        }
                    }
                    request.setAttribute("referenceTranslationLocale", referenceTranslationLocale);
                }
                List<MessageTag> tags = em.createNamedQuery("MessageTag.findAll").getResultList();
                tags.removeAll(message.getMessageTagSet());
                request.setAttribute("unusedTags", tags);
                request.setAttribute("messageKey", request.getParameter("messageKey"));
                request.setAttribute("organizations", userBean.getOrganizations());
                request.setAttribute("allCropCategoryIds", em.createNamedQuery("CropCategory.findByOrganizationId").setParameter("organizationId",user.getOrganizationId().getOrganizationId()).getResultList());
                request.getRequestDispatcher("/messageForm.ftl").forward(request, response);
            }
            catch(NoResultException ex)
            {
                response.sendError(404, "Message not found");
            }
            catch(NullPointerException | NumberFormatException ex)
            {
                response.sendError(500, ExceptionUtil.getStackTrace(ex));
            }
        }
        else if(action.equals("newMessageForm"))
        {
            try
            {
                // Initializing new message, setting default pub and expiry dates
                Message message = new Message();
                message.setDatePub(SystemTime.getSystemTime());
                Calendar cal = Calendar.getInstance();
                cal.setTime(message.getDatePub());
                cal.add(Calendar.MONTH, 3);
                message.setDateValidTo(cal.getTime());
                request.setAttribute("message", message);
                request.setAttribute("messageLocale", 
                        request.getParameter("messageLocale") != null ? 
                                request.getParameter("messageLocale") 
                                : SessionLocaleUtil.getCurrentLocale(request).getLanguage()
                );
                request.setAttribute("unusedTags", em.createNamedQuery("MessageTag.findAll").getResultList());
                request.setAttribute("organizations", userBean.getOrganizations());
                request.setAttribute("allCropCategoryIds", em.createNamedQuery("CropCategory.findByOrganizationId").setParameter("organizationId",user.getOrganizationId().getOrganizationId()).getResultList());
                request.getRequestDispatcher("/messageForm.ftl").forward(request, response);
            }
            catch(NullPointerException | NumberFormatException ex)
            {
                ex.printStackTrace();
                response.sendError(500, ExceptionUtil.getStackTrace(ex));
            }
            catch(Exception ex)
            {
                ex.printStackTrace();
            }
        }
        else if(action.equals("messageFormSubmit"))
        {
            try
            {
                Map<String,String[]> parameterMap;
                List<FileItem> items = null;
                if(JakartaServletFileUpload.isMultipartContent(request))
                {
                    // Create a new file upload handler
                    DiskFileItemFactory dfif = DiskFileItemFactory.builder().get();
                    JakartaServletFileUpload upload = new JakartaServletFileUpload(dfif);

                    // Parse the request
                    items = upload.parseRequest(request);
                    parameterMap = FormUtil.getParameterMap(items,"UTF-8");
                }
                else
                {
                    parameterMap = request.getParameterMap();
                }
                
                // Validate form fields (except uploaded file)
                FormValidation formValidation = FormValidator.validateForm("messageForm", parameterMap,SessionLocaleUtil.getI18nBundle(request),getServletContext());
                
                if(formValidation.isValid())
                {
                    // Store the message
                    Message message = messageBean.storeMessage(formValidation.getFormFields(), user);
                    
                    // New messages should be sent to subscribers
                    if(formValidation.getFormField("messageId").getValueAsInteger() < 0)
                    {
                        messagingBean.sendUniversalMessage(message);
                    }

                    // Delete the current illustration
                    String deleteIllustration = formValidation.getFormField("deleteIllustration").getWebValue();
                    if(deleteIllustration != null && deleteIllustration.equals("true"))
                    {
                        message = messageBean.deleteMessageIllustration(message);
                    }
                    // Store the new illustration (replaces former illustration if not already deleted)
                    if(items != null)
                    {
                        for(FileItem item:items)
                        {
                            if(!item.isFormField() && item.getSize() > 0)
                            {
                                message = messageBean.storeMessageIllustration(message, item);
                            }
                        }
                    }   
                    // If an illustration is attached to the message, update the caption
                    messageBean.updateMessageIllustrationCaption(message, 
                                formValidation.getFormField("messageIllustrationCaptionLocale").getWebValue(),
                                formValidation.getFormField("locale").getWebValue()
                                );
                    
                    
                    // Redirect to form
                    response.sendRedirect(new StringBuilder(Globals.PROTOCOL + "://")
                            .append(ServletUtil.getServerName(request))
                            .append("/message?action=editMessageForm&messageId=").append(message.getMessageId())
                            .append("&messageLocale=").append(formValidation.getFormField("locale").getWebValue())
                            .append("&messageKey=").append("messageUpdated").toString()
                            
                    );
                    
                }
                else
                {
                    // Redirect to form with error messages
                    request.setAttribute("formValidation", formValidation);
                    Integer messageId = formValidation.getFormField("messageId").getValueAsInteger();
                    // Get message, attach it
                    Message message;
                    if(messageId > 0)
                    {
                        message = em.find(Message.class, messageId);
                    }
                    else
                    {
                        message = new Message();
                    }
                    request.setAttribute("message", message);
                    request.setAttribute("messageLocale", 
                            request.getParameter("messageLocale") != null ? 
                                    request.getParameter("messageLocale") 
                                    : SessionLocaleUtil.getCurrentLocale(request).getLanguage()
                    );
                    request.getRequestDispatcher("/messageForm.ftl").forward(request, response);
                }
            }
            catch(NullPointerException | NumberFormatException | FormValidationException | FileUploadException  ex)
            {
                ex.printStackTrace();
                response.sendError(500, ExceptionUtil.getStackTrace(ex));
            }
            catch(Exception ex)
            {
                ex.printStackTrace();
                response.sendError(500, ExceptionUtil.getStackTrace(ex));
            }
        }
        else if(action.equals("deleteMessage"))
        {
            try
            {
                Integer messageId = Integer.valueOf(request.getParameter("messageId"));
                messageBean.deleteMessage(messageId);
                // Redirect to list
                response.sendRedirect(new StringBuilder(Globals.PROTOCOL + "://")
                        .append(ServletUtil.getServerName(request))
                        .append("/message")
                        .append("?messageKey=").append("messageDeleted").toString());
            }
            catch(NullPointerException | NumberFormatException ex)
            {
                
            }
        }
    } 

    // <editor-fold defaultstate="collapsed" desc="HttpServlet methods. Click on the + sign on the left to edit the code.">
    /** 
     * Handles the HTTP <code>GET</code> method.
     * @param request servlet request
     * @param response servlet response
     * @throws ServletException if a servlet-specific error occurs
     * @throws IOException if an I/O error occurs
     */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
    throws ServletException, IOException {
        processRequest(request, response);
    } 

    /** 
     * Handles the HTTP <code>POST</code> method.
     * @param request servlet request
     * @param response servlet response
     * @throws ServletException if a servlet-specific error occurs
     * @throws IOException if an I/O error occurs
     */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
    throws ServletException, IOException {
        processRequest(request, response);
    }

    /** 
     * Returns a short description of the servlet.
     * @return a String containing servlet description
     */
    @Override
    public String getServletInfo() {
        return "Short description";
    }// </editor-fold>

}
